;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; GNU GENERAL PUBLIC LICENSE ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Oligopolios is a model designed for educational purposes as part of PROECEXP
;; Copyright (C) 2007 Jacinto Merino Luis
;; Modified by Segismundo Izquierdo
;; 
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Declaracin de variables ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

globals 
[
  rondas           ;; nmero de rondas transcurridas
  cantidad-total   ;; cantidad total enviada por todos los usuarios al mercado 
  precio-mercado   ;; precio alcanzado en el mercado

  PCP              ;; precio de equilibrio en un mercado de competencia perfecta
  QCP              ;; cantidad de equilibrio en un mercado de competencia perfecta
  PMNP             ;; precio de equilibrio en un mercado de monopolio
  QMNP             ;; cantidad de equilibrio en un mercado de monopolio

  lista-color      ;; lista de colores para asignar a los usuarios
 ]

breed [ students student ]

students-own
[
  user-id              ;; nombre seleccionado por cada usuario
  cantidad-provisional ;; variable auxiliar
  cantidad-venta       ;; cantidad que el usuario desea colocar en el mercado
  Ctotal               ;; costes totales incurridos en la fabricacin de la cantidad-venta  
  beneficios           ;; beneficios obtenidos hasta el momento
  beneficio-ronda      ;; beneficios obtenidos por cada jugador en la ronda en curso
  enviado?             ;; variable booleana que indica si el usuario ha realizado su seleccin en la ronda en curso
 ]

;;;;;;;;;;;;;;;;;;;;;;;;
;; Funciones de Setup ;;
;;;;;;;;;;;;;;;;;;;;;;;;

to startup   ;;inicia la simulacion cuando se abre el programa

  hubnet-set-client-interface "COMPUTER" []
  hubnet-reset

  ;; el plot ser nico para cada usuario.

  __hubnet-make-plot-narrowcast "Mis cantidades"
  __hubnet-make-plot-narrowcast "Mis beneficios"

  ask patches [ set pcolor gray ]
  crear-lista
end

;; reinicia a todos los clientes para comenzar la actividad de nuevo con el mismo grupo

to reiniciar

  set rondas 0
  ask students
    [
      reiniciar-variables
      actualizar-interfaz-clientes
    ]
borrar-graficos
end

;; inicializa las variables de los usuarios

to reiniciar-variables

  hubnet-broadcast "cantidad-al-mercado" 10
  set cantidad-provisional 0
  set cantidad-venta 0
  set Ctotal 0  
  set beneficios 0
  set beneficio-ronda 0
  set enviado? false
  set label ( word  user-id ", " precision beneficios 2 ", " cantidad-venta )

  set cantidad-total 0
  set precio-mercado 0
  set PCP 0
  set QCP 0
  set PMNP 0
  set QMNP 0
end

to borrar-graficos ;; borra los grficos
    
   set-current-plot "Mis beneficios"
   ask students [ __hubnet-clear-plot user-id ]
   
   set-current-plot "Mis cantidades"
   ask students [ __hubnet-clear-plot user-id ]
   
   set-current-plot "Evolucin"
   ask students [ __hubnet-clear-plot user-id ]
   
   set-current-plot "Curva de reaccin"
   ask students [ __hubnet-clear-plot user-id ]

   clear-all-plots  
end

;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Functions ;;
;;;;;;;;;;;;;;;;;;;;;;;

to activar
  every .1
  [ listen-to-clients ] ;; obtiene comandos y datos de los clientes
  
  ;; desarrollo automtico de la sesion, no es necesario pulsar el botn Sgte-ronda
  
   if auto? [ if not any? students with [enviado? = false]  and
               (rondas < num-rondas) [siguiente-ronda]]

end

;; avanza una ronda, se hacen los calculos y se envia la informacin a los clientes

to siguiente-ronda
  set rondas rondas + 1
  ask students [ realizar-calculos ]
  dibujar-graficos
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Codigo para interactuar con los clientes ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 
;; procedimiento standard de hubnet para tomar datos de los clientes 

to listen-to-clients
  while [ hubnet-message-waiting? ]
  [ hubnet-fetch-message
    ifelse hubnet-enter-message?
    [ create-new-student hubnet-message-source ]
    [ ifelse hubnet-exit-message?
      [ remove-student hubnet-message-source ]
      [ execute-command hubnet-message-tag ]
    ]
  ]
end

;; interpretar cada mensaje de hubnet como un comando

to execute-command [command]

 if command = "cantidad-al-mercado" ;; se actualiza la variable cantidad al mercado cada vez que se ajusta la barra 
   [ask students with [user-id = hubnet-message-source]
     [if (enviado? = false)
       [ set cantidad-provisional hubnet-message ]
     ]
   ]

 if command = "Enviar cantidad"
    [ask students with [user-id = hubnet-message-source] ;; cuando se pulsa el boton ENVIAR-CANTIDAD se envia al mercado la cantidad seleccionada en la barra
         [if (enviado? = false)                          ;; Slo se puede enviar productos al mercado una vez por ronda 
             [ set cantidad-venta cantidad-provisional
               hubnet-send user-id "Cantidad actual" precision cantidad-venta 2
               set shape "face happy"
               set enviado? true
             ]
         ]
     ]
  end

;; gestin de la entrada y salida de usuarios 

to create-new-student [ id ]
  let p one-of patches with [ count neighbors = 8 and not any? turtles-here and 
                                     not any? neighbors4 with [ any? turtles-here ] ]
  ifelse p = nobody
  [ user-message "Todos los espacios en esta vista estan completos, no se puede agregar ms usuarios." 
    stop]
  [ create-students 1
    [ set user-id  id
      set shape "factory"
      set color  item 0 lista-color
      set lista-color but-first lista-color
      set label-color black
      setxy [pxcor] of p + 0.5 [pycor] of p
      hubnet-send user-id  "Jugador" user-id
        reiniciar-variables
        actualizar-interfaz-clientes
      ]
  ]
end

to remove-student [ id ]  ;; eliminar usuarios que se desconecten

  ask students with [user-id = id] [ die ]
  
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Procedimiento principal ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to realizar-calculos  ;; se realizan los calculos 

  set Ctotal (coste-fijo + (A * cantidad-venta) + (B * (cantidad-venta ^ 2)))
  set cantidad-total  sum [cantidad-venta] of students
  set precio-mercado  ((Demanda-base - cantidad-total) / (Sensibilidad-demanda))
  if  precio-mercado  < 0 [set precio-mercado 0 ] 
                             
  set beneficio-ronda (precio-mercado * cantidad-venta) - Ctotal
  set beneficios beneficios + beneficio-ronda
  set label ( word  user-id ", " precision beneficios 2 ", " cantidad-venta )
  set enviado? false

  ;; resultados esperados en un mercado de competenia perfecta o monopolio segun la teora econmica

      set PCP   (((2 * B * Demanda-base) + (count students * A)) / (count students + (2 * B * Sensibilidad-demanda)))
      set QCP   Demanda-base - (Sensibilidad-demanda * PCP)

      set QMNP  (( count students *( Demanda-base - (Sensibilidad-demanda * A))) / ((2 * B * Sensibilidad-demanda) + (2 * count students)))
      set PMNP  ((Demanda-base - QMNP) / (Sensibilidad-demanda))


  realizar-graficos
  actualizar-interfaz-clientes

end

;; se envia la informacion de vuelta a la interfaz del usuario 

to actualizar-interfaz-clientes

  set shape "factory"
  hubnet-send user-id "Jugador" user-id
  hubnet-send user-id "Ronda" rondas
  hubnet-send user-id "Beneficios acumulados" precision beneficios 2
  hubnet-send user-id "Beneficios ltima ronda" precision beneficio-ronda 2
  hubnet-send user-id "Cantidad actual" precision cantidad-venta 2
  hubnet-send user-id "ltimo Precio" precision precio-mercado 2
  hubnet-send user-id "Coste produccin" precision Ctotal 2
  hubnet-send user-id "Cantidad resto" precision (cantidad-total - cantidad-venta) 2
  enviar

end

to enviar 

  hubnet-broadcast "Funcin de costes" (word "Ctotal = " coste-fijo " + " A " *Q"  " + "  B  " *Q^2")

end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Cdigo para realizar los grficos ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to dibujar-graficos
   set-current-plot "Mercado"
   clear-plot

    ;; dibuja hasta que la demanda cruza el eje x (cuando el precio de la demanda es 0)

   let Q 0   
    while [ ((Demanda-base - Q)/  Sensibilidad-demanda) >= 0 ] 
  [
      ;; Dibujar la Demanda

      set-current-plot-pen "Demanda"
      plot ((Demanda-base - Q)/  Sensibilidad-demanda)

      ;; Dibujar Ingreso Marginal

      set-current-plot-pen "IMa"
      if    (Demanda-base - 2 * Q) / Sensibilidad-demanda >= 0 
      [plot (Demanda-base - 2 * Q) / Sensibilidad-demanda]

      set Q (Q + 1 )
  ]
  
   set-plot-x-range 0 Q
   set-current-plot "Mercado"

    ;; Dibujar la Oferta

   let X 0
   
   ;; while [X  <= Demanda-base / Sensibilidad-demanda ] ;; nos marca hasta cuando dibuja la curva de oferta
    while [X  <= Demanda-base ]
    [set-current-plot-pen "Oferta"
      plotxy  X (A + 2 * B * (X / count students))
      set X (X + 1 )
    ]
 ;; Dibujar equilibrio del mercado

 let Y -10
     while [Y < 21]
     [set-current-plot-pen "mercado"
      plotxy (cantidad-total) (precio-mercado + Y)
      plotxy (cantidad-total + Y) (precio-mercado)
      set Y (Y + 1 )]

  set-current-plot "Evolucin"
  set-current-plot-pen "Qtotal"
  plotxy   rondas  cantidad-total

  set-current-plot-pen "Precio"
  plotxy   rondas precio-mercado

  ;; Plots especficos para cada cliente 

 set-current-plot "Mis cantidades"
 ask students [ __hubnet-plotxy user-id rondas cantidad-venta]

 set-current-plot "Mis beneficios"
 ask students [ __hubnet-plotxy user-id rondas beneficio-ronda]


  set-current-plot "Curva de reaccin"
  clear-plot

    let Z 0
    while [((Demanda-base - Z - (Sensibilidad-demanda * A)) / (2 + (2 * B * Sensibilidad-demanda))) >= 0] [
    set-current-plot-pen "C reaccin"
    plotxy  ((Demanda-base - Z - (Sensibilidad-demanda * A)) / (2 + (2 * B * Sensibilidad-demanda))) Z

    set Z (Z + 10)
    ]

end

to realizar-graficos

 ;; dibuja las cantidades enviadas por cada jugador en un mismo grfico
  set-current-plot "Cantidades"
  create-temporary-plot-pen user-id
  set-plot-pen-color color 
  plotxy rondas cantidad-venta
  
  ;; dibuja el beneficio obtenido en cada ronda por cada jugador en un mismo grfico
  set-current-plot "Beneficios"
  create-temporary-plot-pen user-id
  set-plot-pen-color color 
  plotxy rondas beneficio-ronda

end 

to crear-lista

  ;; prepara los colores para los clientes
  ;; NOTA solamente podr haber tantos clientes como colores en la lista 
  ;; en la lista actual hay 33 colores, suficientes para cualquier experiencia.
  set lista-color   (list red orange pink lime sky magenta
                          cyan turquoise green  brown blue
                          red orange pink lime sky magenta
                          cyan turquoise green  brown blue
                          red orange pink lime sky magenta
                          cyan turquoise green  brown blue)
end


;; The following copyright is included because the coding of some procedures 
;; was inspired from models that included this copyright.

; *** NetLogo Model Copyright Notice ***
;
; This model was created as part of the projects:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN
; CLASSROOMS and INTEGRATED SIMULATION AND MODELING ENVIRONMENT.
; The project gratefully acknowllinks the support of the
; National Science Foundation (REPP & ROLE programs) -- grant numbers
; REC #9814682 and REC-0126227.
;
; Copyright 2005 by Uri Wilensky.  Updated 2005.  All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from Uri Wilensky.
; Contact Uri Wilensky for appropriate licenses for redistribution for
; profit.
;
; To refer to this model in academic publications, please use:
; Wilensky, U. (2005).  NetLogo.
; http://ccl.northwestern.edu/netlogo/models/PreferentialAttachment.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.
;
; In other publications, please use:
; Copyright 2005 by Uri Wilensky.  All rights reserved.  See
; http://ccl.northwestern.edu/netlogo/models/PreferentialAttachment
; for terms of use.
;
; *** End of NetLogo Model Copyright Notice ***
@#$#@#$#@
GRAPHICS-WINDOW
695
11
1241
330
6
3
41.231
1
12
1
1
1
0
0
0
1
-6
6
-3
3
0
0
1
ticks

BUTTON
125
11
220
44
Activar
activar
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
23
350
227
383
A
A
0
200
100
0.1
1
NIL
HORIZONTAL

BUTTON
125
49
220
82
Sgte ronda
siguiente-ronda
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

MONITOR
11
94
69
139
Ronda
rondas
0
1
11

BUTTON
26
10
121
43
Reiniciar
reiniciar
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
23
308
227
341
coste-fijo
coste-fijo
0
500
0
1
1
unidades
HORIZONTAL

MONITOR
77
94
179
139
Cantidad total
cantidad-total
2
1
11

MONITOR
184
94
243
139
Precio
precio-mercado
2
1
11

PLOT
257
226
471
425
Evolucin
Rondas
Precio - Q total
0.0
10.0
0.0
10.0
true
true
PENS
"Precio" 1.0 0 -13791810 true
"Qtotal" 1.0 0 -2674135 true

TEXTBOX
26
244
231
308
El coste total para cada usuario viene definido como:\n\nCtotal = CF + (A  Q) + (B  Q^2)
11
0.0
0

PLOT
256
10
688
220
Mercado
Cantidad
Precio
0.0
5.0
0.0
5.0
true
true
PENS
"Demanda" 1.0 0 -8630108 true
"IMa" 1.0 0 -6459832 true
"Oferta" 1.0 0 -16777216 true
"mercado" 1.0 2 -2674135 true

SLIDER
21
449
225
482
Demanda-base
Demanda-base
0
1000
1000
1
1
NIL
HORIZONTAL

SLIDER
21
494
225
527
Sensibilidad-demanda
Sensibilidad-demanda
0
5
2.01
0.01
1
NIL
HORIZONTAL

TEXTBOX
27
541
301
601
La Demanda viene dada por:\n\nQ = Demanda-base - ( Sensibilidad-demanda  P )  
11
0.0
0

MONITOR
504
447
581
492
Precio CP
pcp
3
1
11

MONITOR
505
505
582
550
Precio MNP
pmnp
3
1
11

MONITOR
591
505
685
550
Cantidad MNP
qmnp
3
1
11

MONITOR
590
447
684
492
Cantidad CP
QCP
3
1
11

TEXTBOX
280
453
497
507
Se muestran los resultados tericos que proporcionan los modelos econmicos para un mercado que funcione en Competencia Perfecta o en Monopolio
11
0.0
0

SLIDER
23
394
228
427
B
B
0
5
0
0.01
1
NIL
HORIZONTAL

PLOT
753
352
913
472
Mis cantidades
NIL
NIL
0.0
10.0
0.0
10.0
true
false
PENS
"cantidad" 1.0 0 -16777216 true

PLOT
698
342
967
557
Cantidades
Rondas
Cantidades
0.0
10.0
0.0
10.0
true
false
PENS
"default" 1.0 0 -16777216 true

BUTTON
24
49
120
82
Enviar info
enviar
NIL
1
T
TURTLE
NIL
NIL
NIL
NIL

PLOT
1009
374
1169
494
Mis beneficios
NIL
NIL
0.0
10.0
0.0
10.0
true
true
PENS
"Beneficio" 1.0 0 -16777216 true

PLOT
972
343
1242
558
Beneficios
Rondas
Beneficios
0.0
10.0
0.0
10.0
true
false
PENS
"default" 1.0 0 -16777216 true

PLOT
475
228
689
425
Curva de reaccin
Q1
Qtotal - Q1
0.0
10.0
0.0
10.0
true
false
PENS
"C reaccin" 1.0 0 -5825686 true

SLIDER
49
191
221
224
num-rondas
num-rondas
0
50
30
1
1
rondas
HORIZONTAL

SWITCH
79
151
182
184
auto?
auto?
0
1
-1000

@#$#@#$#@
QUE ES ESTO?
-----------
Esta es una actividad diseada para simular el modelo de Cournot de competencia en cantidades. El modelo de Cournot fue formulado en 1838 y trata de explicar el comportamiento de las empresas que se encuentran en un mercado de competencia imperfecta, en concreto en una situacin de oligopolio. Este tipo de mercado se caracteriza por contar con pocos competidores, que ofertan productos que son muy buenos sustitutivos entre s. Las decisiones sobre cantidades y precios tomadas por los competidores han de ser tenidas muy en cuenta a la hora de tomar las propias decisiones.
Esta actividad est pensada en principio para duopolios pudindose extender a ms de dos competidores, en el cual los usuarios deciden fabricar y enviar una cierta cantidad de producto al mercado y el programa, segn la funcin de demanda del mercado, calcula el precio que alcanza la cantidad total enviada por todos los usuarios, as como los beneficios para los productores.

NOVEDADES
-----------

ESTA NUEVA VERSION DE OLIGOPOLIOS NOS PERMITE DESARROLLAR LAS SESIONES DE FORMA AUTOMTICA. ESTA NUEVA CARACTERISTICA RESULTA DE GRAN AYUDA A LA HORA DE LLEVAR A CABO VARIAS SESIONES A LA VEZ, YA QUE NO ES NECESARIO QUE EL DOCENTE EST PENDIENTE DE CUANDO HAN DECIDIDO TODOS LOS USUARIOS PARA PULSAR EL BOTN Sgte ronda, YA QUE EL PROGRAMA LO REALIZA DE MANERA AUTOMTICA.

Novedades

auto?: si el interruptor est en ON la sesin se lleva a cabo de forma automtica, de forma que las rondas avanzarn de forma automtica en cuanto todos los usuarios hayan realizado su decisin acerca de la cantidad que envian al mercado. La duracin de la sesin se determina mediante la barra num-rondas.
Si est en OFF el docente deber pulsar el botn Sgte-ronda para que avance el desarrollo de la sesin.

num-rondas: determina el nmero de rondas que dura la sesin, en caso de que el interruptor auto? est en ON. El nmero de rondas se ha limitado a 30, pero en caso de desear desarrollar una sesin con un numero mayor de rondas bastar con editar la barra num-rondas.


CMO FUNCIONA?
-----------------

Los usuarios deciden que cantidad de producto van a fabricar y enviar al mercado.
Cada vendedor dispone de una funcin de costes de su proceso productivo (la misma para todos).
Los vendedores deciden el nmero de unidades que fabricarn y enviarn al mercado. La decisin la tomarn teniendo en cuenta la funcin de costes que les ha sido asignada.
El mercado absorbe toda la oferta de todos los vendedores y devuelve un precio segn la funcin de demanda del mercado.
El beneficio econmico para el vendedor es la diferencia entre el precio (cobrado) y el coste de produccin.

CMO USARLO?
-------------

Botones

Reiniciar: al pulsar este botn el programa reinicia todas las variables de los usuarios y las rondas para iniciar una nueva sesin con el mismo grupo. 

Activar: pone en marcha la simulacin. Este botn es de tipo permanente (forever), es decir, una vez pulsado permanece activado hasta que no se vuelve a pulsar. 

Enviar info: este botn debe pulsarse en caso de que se varen las condiciones de costes durante la simulacin, de modo que los usuarios estn informados de ello y puedan utilizar esta informacin para tomar sus decisiones.

Sgte ronda: al pulsar este botn el programa hace avanzar las rondas, realiza los clculos con los datos obtenidos de los clientes, y muestra los resultados a los usuarios y al servidor.

Pantallas

Ronda: nos indica el nmero de rondas transcurridas, las rondas avanzan al pulsar el botn Sgte ronda.

Cantidad total: muestra la cantidad total ofertada en el mercado por todos los vendedores en la ronda anterior.

Precio: precio de equilibrio alcanzado en el mercado para la cantidad total ofertada por los vendedores. Es la retribucin que se pagar a los vendedores.

Precio CP: marca el precio terico de equilibrio (el que maximiza el beneficio) que se alcanzara en un mercado de competencia perfecta para las condiciones de demanda y costes dados.

Cantidad CP: marca la cantidad terica (la cantidad que maximiza el beneficio) que se vendera en un mercado de competencia perfecta para las condiciones de demanda y costes dados en el equilibrio.

Precio MNP: marca el precio que se alcanzara en un mercado en el que los vendedores se comportaran como monopolistas, es decir como si hubiera un solo productor.

Cantidad MNP: indica la cantidad que se vendera en caso de encontrarnos en una situacin de monopolio.

Barras

Coste fijo: ajusta el coste fijo por ronda para los usuarios.

A: ajusta el factor de costes de los usuarios.

B: ajusta el factor cuadrtico de los costes. (Por cuestiones de funcionamiento siempre debe ser B > 0, en caso contrario aparecer un mensaje de error)

Coste total = coste-fijo + A  Q + B  Q^2

Demanda-base: ajusta la demanda base para la simulacin.

Sensibilidad-demanda: ajusta la sensibilidad de la demanda al precio.

Q = Demanda-base  Sensibilidad-demanda  P 

Grficos

El grfico Mercado refleja la situacin del mercado mediante la representacin de una serie de parmetros.
Demanda: dibuja la curva de demanda del mercado determinada por la funcin de demanda previamente definida mediante las barras Demanda-base y Sensibilidad-demanda.

IMa: representa el ingreso marginal, es decir el aumento de ingresos que provoca la venta de una unidad adicional de producto.

Oferta: refleja la oferta de la industria. Es la suma horizontal de los costes marginales de cada una de las empresas que acta en el mercado. Donde corta a la demanda sera la solucin terica de un modelo de competencia perfecta.

Mercado: la cruz representa la solucin de la ronda anterior en funcin de la oferta total de los oligopolistas en el mercado.

Evolucin: este grfico refleja la evolucin de los precios del mercado y de la cantidad total ofertada a lo largo de las rondas.

Curva de reaccin: refleja la curva de reaccin de cualquiera de las empresas que operan en el mercado frente al resto. La forma de leer este grfico es la siguiente: entraramos con la cantidad total ofertada por el resto de competidores por el eje Y (Qtotal  Q1) hasta cortar la curva de reaccin. Despus bajando paralelamente al eje Y hasta alcanzar el eje X (Q1). Ese punto representa la cantidad que deberamos producir para maximizar nuestros beneficios, siempre y cuando nuestros competidores no varen la decisin que tomaron en la ronda anterior.

Cantidades: refleja las cantidades enviadas por cada uno de los usuarios en cada ronda, en un mismo grfico. 

Beneficios: refleja el beneficio obtenido por cada uno de los usuarios en cada ronda, en un mismo grfico. 

Interfaz del usuario

Jugador: indica el nombre que el usuario ha escogido al conectarse a la sesin.

Ronda: indica el nmero de rondas transcurridas.

Funcin de costes: funcin de costes para el productor. Informa al usuario de la funcin de costes de su proceso productivo seleccionada en la interfaz del servidor.

Cantidad actual: en esta pantalla aparecer la cantidad que en el momento de pulsar el botn Enviar cantidad indicaba la barra cantidad-al-mercado. Refleja la cantidad que para la ronda en curso el usuario ha decidido fabricar y poner en el mercado para su venta.

Cantidad-al-mercado: barra mediante la cual los usuarios seleccionan la cantidad que desean producir y enviar al mercado. 

Enviar cantidad: al presionar este botn queda fijada la cantidad que en ese momento indica la barra cantidad-al-mercado. Esa cantidad se enva al mercado para su venta. Una vez pulsado, la seleccin no se puede cambiar hasta la siguiente ronda.

Cantidad resto: nos indica la cantidad total que se envi al mercado en la ronda anterior, sin contar lo que enviamos nosotros. Es decir, lo que enviaron el resto de productores.

Ultimo precio: indica el precio que alcanz en el mercado la cantidad total enviada por todos los usuarios en la ronda anterior, incluida la cantidad enviada por el usuario.

Coste produccin: costes incurridos por el usuario en la produccin de la cantidad enviada al mercado en la ronda anterior.

Beneficios ltima ronda: beneficios obtenidos con la venta de la cantidad enviada al mercado en la ltima ronda.

Beneficios acumulados: beneficios obtenidos a lo largo de las rondas.

Evolucin: en este grfico podemos ver reflejada la evolucin de la cantidad total (Qtotal) y del precio alcanzado en el mercado (Precio) a lo largo de las rondas.

Curva de reaccin: este grfico dibuja la curva de reaccin del usuario. Sirve de ayuda para decidir la mejor opcin para la siguiente jugada siempre que el resto no varen su estrategia hacemos lo mejor posible.

Mis cantidades: en este grfico se reflejan las cantidades enviadas al mercado por el usuario en las sucesivas rondas.

Mis beneficios: en este grfico se reflejan los beneficios por ronda obtenidos por el usuario. No refleja lo beneficios acumulados sino los beneficios obtenidos en cada ronda.



PUESTA EN MARCHA 
----------------

1. Abrir la actividad "Oligopolios".
2. Ajustar el ZOOM si es necesario.
3. Pulsar ACTIVAR.
4. Pedir a los usuarios que abran el HubNet Client y se conecten a la sesin.
5. Ajustar los valores de las barras COSTE-FIJO, A, B, DEMANDA-BASE y SENSIBILIDAD-DEMANDA
6. Pulsar REINICIAR.
7. Indicar a los usuarios que realicen su seleccin de CANTIDAD y pulsen ENVIAR SELECCIN.
8. Cuando todos hayan realizado su seleccin (cambian los iconos en la vista 2D) pulsar el boton SGTE RONDA.
9. Repetir el punto 7 y 8 tantas veces como se desee.
10. Al termino de la sesin analizar los resultados con el grupo.


MODELOS RELACIONADOS
--------------
Subasta doble, Tragedy of the Commons HubNet, Oil Cartel, Public Good, Restaurants


CREDITS AND REFERENCES
----------------------

Este modelo ha sido creado por Jacinto Merino Luis (2007) 

NetLogo credits
Wilensky, U. (1999) NetLogo. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

factory
false
0
Rectangle -7500403 true true 76 194 285 270
Rectangle -7500403 true true 36 95 59 231
Rectangle -16777216 true false 90 210 270 240
Line -7500403 true 90 195 90 255
Line -7500403 true 120 195 120 255
Line -7500403 true 150 195 150 240
Line -7500403 true 180 195 180 255
Line -7500403 true 210 210 210 240
Line -7500403 true 240 210 240 240
Line -7500403 true 90 225 270 225
Circle -1 true false 37 73 32
Circle -1 true false 55 38 54
Circle -1 true false 96 21 42
Circle -1 true false 105 40 32
Circle -1 true false 129 19 42
Rectangle -7500403 true true 14 228 78 270

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
VIEW
404
509
560
593
0
0
0
1
1
1
1
1
0
1
1
1
-6
6
-3
3

MONITOR
40
13
202
62
Jugador
NIL
3
1

MONITOR
205
13
265
62
Ronda
NIL
3
1

MONITOR
293
173
496
222
Beneficios acumulados
NIL
3
1

SLIDER
37
178
263
211
cantidad-al-mercado
cantidad-al-mercado
0
500
5
1
1
unidades
HORIZONTAL

MONITOR
93
124
205
173
Cantidad actual
NIL
3
1

MONITOR
407
14
495
63
ltimo Precio
NIL
3
1

BUTTON
90
220
218
253
Enviar cantidad
NIL
NIL
1
T
OBSERVER
NIL
NIL

MONITOR
293
120
496
169
Beneficios ltima ronda
NIL
3
1

PLOT
6
262
256
463
Evolucin
Rondas
Precio / cantidad 
0.0
10.0
0.0
10.0
true
true
PENS
"Precio" 1.0 0 -13791810 true
"Qtotal" 1.0 0 -2674135 true

PLOT
516
262
766
463
Mis cantidades
Rondas
Cantidades
0.0
10.0
0.0
10.0
true
true
PENS
"cantidad" 1.0 0 -16777216 true

MONITOR
40
68
266
117
Funcin de costes
NIL
3
1

PLOT
770
262
1020
463
Mis beneficios
Rondas
Beenficios
0.0
10.0
0.0
10.0
true
true
PENS
"Beneficio" 1.0 0 -16777216 true

PLOT
261
262
511
463
Curva de reaccin
Q1
Qtotal - Q1
0.0
10.0
0.0
10.0
true
true
PENS
"C reaccin" 1.0 0 -5825686 true

MONITOR
293
69
496
118
Coste produccin
NIL
3
1

MONITOR
293
13
399
62
Cantidad resto
NIL
2
1

PLOT
769
34
1021
252
Beneficios
Rondas
Beneficios
0.0
10.0
0.0
10.0
true
false
PENS
"default" 1.0 0 -16777216 true

PLOT
511
31
763
250
Cantidades
Rondas
Cantidades
0.0
10.0
0.0
10.0
true
false
PENS
"default" 1.0 0 -16777216 true

@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
